package furny.ga.logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.TypedQuery;

import furny.ga.logger.entities.EvaluationRunEntry;
import furny.ga.logger.entities.EventType;
import furny.ga.logger.entities.GeneEntry;
import furny.ga.logger.entities.IndividualEntry;
import furny.ga.logger.entities.LogEvent;

/**
 * Database manager for statistics.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class StatisticsDBManager {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(StatisticsDBManager.class.getName());

  private static final StatisticsDBManager INSTANCE = new StatisticsDBManager();

  private static final String PARAMETERS = ";PAGE_SIZE=8192;AUTO_SERVER=TRUE;MVCC=TRUE;LOCK_TIMEOUT=100;DB_CLOSE_DELAY=-1;CACHE_SIZE=131072";

  private EntityManagerFactory entityManagerFactory;

  /**
   * Private constructor for single instance.
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  private StatisticsDBManager() {
  }

  /**
   * Getter for the single instance.
   * 
   * @return Single instance of this manager.
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  public static StatisticsDBManager getInstance() {
    return INSTANCE;
  }

  /**
   * Creates an {@link EntityManager} to start a database transaction.
   * 
   * @return A new entitity manager.
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  private EntityManager createEntityManager() {
    if (entityManagerFactory == null) {
      entityManagerFactory = Persistence.createEntityManagerFactory("furny");
    }

    return entityManagerFactory.createEntityManager();
  }

  /**
   * Sets a custom file name for the database.
   * 
   * @param filename
   *          Statistics database file name.
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  public void setCustomFilename(final String filename) {
    if (filename != null) {
      if (entityManagerFactory != null) {
        entityManagerFactory.close();
      }

      final Map<String, Object> dbProps = new HashMap<String, Object>();
      dbProps.put("javax.persistence.jdbc.url", "jdbc:h2:" + filename
          + PARAMETERS);

      entityManagerFactory = Persistence.createEntityManagerFactory(
          "statistics", dbProps);
    }
  }

  // DATABASE ACCESSOR METHODS
  /**
   * Saves a run.
   * 
   * @param run
   *          The evaluation run.
   * @param merge
   *          Merge (true) or create.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void saveEvaluationRun(final EvaluationRunEntry run,
      final boolean merge) {
    final EntityManager entityManager = createEntityManager();

    EvaluationRunEntry er = run;
    try {
      entityManager.getTransaction().begin();

      if (merge) {
        er = entityManager.merge(er);
      } else {
        entityManager.persist(er);
      }

      entityManager.getTransaction().commit();

      // fireFurnitureUpdate(f.getId(), f);
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to save evaluation run", e);

      entityManager.getTransaction().rollback();
    } finally {
      entityManager.close();
    }
  }

  /**
   * Save a new log event.
   * 
   * @param event
   *          The event.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void saveLogEvent(final LogEvent event) {
    final EntityManager entityManager = createEntityManager();

    try {
      entityManager.getTransaction().begin();

      entityManager.persist(event);

      entityManager.getTransaction().commit();

      // fireFurnitureUpdate(f.getId(), f);
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to save log event", e);

      entityManager.getTransaction().rollback();
    } finally {
      entityManager.close();
    }
  }

  /**
   * Save a gene entry.
   * 
   * @param gene
   *          The gene entry.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void saveGeneEntry(final GeneEntry gene) {
    final EntityManager entityManager = createEntityManager();

    try {
      entityManager.getTransaction().begin();

      entityManager.persist(gene);

      entityManager.getTransaction().commit();
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to save gene entry", e);

      entityManager.getTransaction().rollback();
    } finally {
      entityManager.close();
    }
  }

  /**
   * Save a individual entry.
   * 
   * @param individual
   *          The entry.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void saveIndividualEntry(final IndividualEntry individual) {
    final EntityManager entityManager = createEntityManager();

    try {
      entityManager.getTransaction().begin();

      individual.setEvent(entityManager.merge(individual.getEvent()));

      for (final GeneEntry entry : individual.getGenes()) {
        entityManager.persist(entry);
      }

      entityManager.persist(individual);

      entityManager.getTransaction().commit();
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to save gene entry", e);

      entityManager.getTransaction().rollback();
    } finally {
      entityManager.close();
    }
  }

  /**
   * Get all run entries.
   * 
   * @return Entries of evaluation runs.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public List<Long> getRunEntries() {
    final EntityManager entityManager = createEntityManager();

    List<Long> result = null;
    try {
      entityManager.getTransaction().begin();

      final TypedQuery<Long> q = entityManager.createQuery(
          "SELECT id FROM EvaluationRunEntry", Long.class);

      q.setHint("org.hibernate.cacheable", true);

      result = q.getResultList();

      entityManager.getTransaction().commit();
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to get furniture ids", e);

      entityManager.getTransaction().rollback();

      result = new ArrayList<Long>();
    } finally {
      entityManager.close();
    }

    return result;
  }

  /**
   * Get individuals by their types.
   * 
   * @param runId
   *          Id of the evaluation run.
   * @param types
   *          Types of individuals.
   * @return List of individual entries.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public List<IndividualEntry> getIndividuals(final Long runId,
      final EventType... types) {
    final EntityManager entityManager = createEntityManager();

    List<IndividualEntry> result = null;
    try {
      final StringBuilder sb = new StringBuilder(
          "FROM IndividualEntry i WHERE i.event.evaluationRun.id=:id");

      if (types.length > 0) {
        sb.append(" AND (");
        for (int i = 0; i < types.length; i++) {
          if (i > 0) {
            sb.append(" OR ");
          }
          sb.append("i.event.type=:type" + i);

          if (i == types.length - 1) {
            sb.append(')');
          }
        }
      }

      entityManager.getTransaction().begin();
      final TypedQuery<IndividualEntry> q = entityManager.createQuery(
          sb.toString(), IndividualEntry.class);
      q.setParameter("id", runId);

      for (int i = 0; i < types.length; i++) {
        q.setParameter("type" + i, types[i]);
      }

      q.setHint("org.hibernate.cacheable", true);

      result = q.getResultList();

      entityManager.getTransaction().commit();
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to get furniture", e);

      entityManager.getTransaction().rollback();
    } finally {
      entityManager.close();
    }

    return result;
  }

  /**
   * Lets the manager exit and clean up its resources. It will not be reusable
   * after that!
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  public void exit() {
    entityManagerFactory.close();
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          arguments
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    System.err.println(getInstance().getIndividuals(3L,
        EventType.POPULATION_INITIATED, EventType.INDIVIDUALS_INSERTED));
  }
}
